#include "../../system.ped"
#include "../../messagebox.ped"
#include "../../keycodes.ped"
#include "../../Controls/preset.ped"

class App@Scope@ModuleMIDIControllerList extends BasicList
{
	public object	SelDevice;
    object          SelMIDIManager; //the current selected MIDIManager (which can be seen as the current MIDI Port)
    public int      MIDIManager;    //the index into the systems MIDIManagerArray
	public int		Freeze=0;       //Freeze current selected Device/Module

	public string	ModuleName;     //The Devices/Modules name
	public int		SortColumn;
    public MIDI     MIn;            //MIDI Input, automatically routed, when "MIDIManager" changes
    public int      ShowNames;      //if 1, shows appropriate controller names, instead of controller numbers (0)
    public int      LearnMode;      //In this mode, controllers from the selected Port can be learned

    public int      CType;          //These 4 pads reflect the controllers ID of the selected parameter,
    public int      CCtrl1;         //or, if LearnMode==1, reflect the controllers ID of the learned controller
    public int      CCtrl2;
    public int      CChannel;

    public int      ParameterIndex; //contains the current index of a controllable parameter, which is located
    object          iParamList[];   //in the iParamList-Array. iParamCount specifies the number of parameters
    int             iParamCount;    //in the array. However, ParameterIndex can be equal to iParamCount, to specify
                                    //the "No Parameter" index. 

    public int      NoteBase;       //determines, how noteoff/on and keypressure controllers names look like
                                    //e.g. -2 means, that NoteOn with note number 0 will be displayed as "C-2"
                                    //just in case, ShowNames == 1

    public int      AddController;  //whenever this value changes, the current settings are taken, to assign
                                    //a new controller, if possible.
    public int      RemController;  //whenever this value changes, the current selected controller assignment
                                    //will be removed.

    public int      FixedChannel;   //reflects the state of the current restored preset, wether it allows
                                    //to automatically change the midi channel of all controllers when
                                    //the device midi channel changes (0) or not (1)
    public int      MidiChannel;

    object			_TTParameter;
	object			_TTNormal;
	object			_TTCheckBox;
	object			_BTCheckBox;
	object			_BTCheckBoxOK;

	object			CurrentDevice;  //specifies the last selected Module/Device which was selected during the
                                    //state "Freeze == 0"

    object          CtrlList1;
    object          CtrlList2;
    object          DevParamList;
    object          ManagerList;
    object          NoteList;

    object          curParameter;
    object          ListVI;

#define COL_PARAMETER   		0
#define COL_MIDICHANNEL   		1
#define COL_MIDICTRLTYPE		2
#define COL_MIDICTRL1   		3
#define COL_MIDICTRL2       	4

	static {
		_TTNormal = GOByName("TTNormal");
        _TTParameter = GOByName( "TTParameter" );
		_BTCheckBox = GOByName( "BTCheckBox" );
		_BTCheckBoxOK = _BTCheckBox.ChildByName( "OK" );
		InitCols();
		AdvancedMessaging(true);
        CallMeOnDelete(true);
        NoteBase = -2;
        SApp.SetMidiAssigner(this);
	}


	public void OnPrepareModuleDeleting() 
	{
		SApp.SetMidiAssigner(nil);
	}

    int GetChannel(int v){
        return (v>>16)&0xf;
    }
    
    int GetCType(int v){
        return (v>>20)-8;
    }

    int GetCNum(int v){
        return (v>>8)&0xff;
    }

    int SetChannel(int orig,int newch){
        return (orig&0x7ff0ffff)|(newch<<16);
    }
    
    int SetCType(int orig,int newType){
        return (orig&0x7f0fffff)|((newType+8)<<20);
    }

    int SetCNum(int orig,int newcn){
        return (orig&0x7fff00ff)|(newcn<<8);
    }

    int GetAllManagers(){
        object managers[];
        int m = SApp.GetMidiManagers(managers);
        count(ManagerList.Entries,m);
        count(ManagerList.Values,m);
        int retVal = -1;
        int i;
        for(i=0;i<m;i++){
            ManagerList.Entries[i] = managers[i].Parent().Name();
            ManagerList.Values[i] = i;
            if(SelMIDIManager == managers[i]) retVal = i;
        }
        ManagerList.OnEntriesChanged();
        return retVal;
    }
    
    string GetControllerString(int ctype,int cnum){
        if(ShowNames != 0){
            switch(ctype){
                case 0:
                case 1:
                case 2:
                    return NoteList.Entries[cnum];
                case 3:
                    return CtrlList2.ChildByIndex(0).Entries[cnum];
                default:
                    return CtrlList1.ChildByIndex(0).Entries[cnum];
            }
        }
        else{
            return format("%03i",cnum);
        }
    }

    void SetCPadsFromController(object ctrl){
        int val = ctrl.GetCtrl1();
        CType = GetCType(val);
        CChannel = GetChannel(val);
        if(CType<5){
            CCtrl1 = GetCNum(val);
            if(ctrl.Is14Bit())
                CCtrl2 = GetCNum(ctrl.GetCtrl2());
            else
                CCtrl2 = 128;
        }
        else{
            CCtrl1 = 128;
            CCtrl2 = 128;
        }
    }

    void SetCPadsFromValue(int val){
        CType = GetCType(val);
        CChannel = GetChannel(val);
        if(CType<5){
            CCtrl1 = GetCNum(val);
            CCtrl2 = 128;
        }
        else{
            CCtrl1 = 128;
            CCtrl2 = 128;
        }
    }

    void SetDisplayFromParameter(int paridx){
        object px = GetIParameter(paridx);  //do we have a parameter at index 'paridx' ?
        object item = FindPData(nil,px,0);  //and find the corresponding item
        if(item != nil){
            //if we found an entry
            SelectItem(item,0);         //DOES NOT CALL OnSelectionChanged, what a LUCK !
            ScrollToItem(item);
            object ctrl = SelMIDIManager.ControllerByParameter(px);
            if(ctrl != nil){
                SetCPadsFromController(ctrl);
                ChangeListNames();
                VarByName("CChannel").Send();
                VarByName("CType").Send();
                VarByName("CCtrl1").Send();
                VarByName("CCtrl2").Send();
            }
            else{
                if(LearnMode == 0){
                    VarByName("CChannel").ValueSend(16);
                    VarByName("CType").ValueSend(8);
                    VarByName("CCtrl1").ValueSend(128);
                }
                VarByName("CCtrl2").ValueSend(128);
            }
        }
        else{
            DeselectItem();
            ScrollToItem(GetItemByNumber(0));
            if(LearnMode == 0){
                VarByName("CChannel").ValueSend(16);
                VarByName("CType").ValueSend(8);
                VarByName("CCtrl1").ValueSend(128);
            }
            VarByName("CCtrl2").ValueSend(128);
        }
    }


    int GetInternalParamIdx(object item){
        object p;
        if(item == nil) return iParamCount;
        GetItemPData(item,0,p);
        for(int i=0;i<iParamCount;i++)
            if(iParamList[i] == p) return i;
        return iParamCount;
    }

    object GetIParameter(int internalIdx){
        if(internalIdx < iParamCount)
            return iParamList[internalIdx];
        else
            return nil;
    }

    void ChangeListNames(){
        if(ShowNames != 0){
            switch(CType){
                case 0: //NoteOff
                case 1: //NoteOn
                case 2: //KeyPressure
                    object p1 = NoteList;
                    CtrlList1.PadByName("List").Var().Value(p1);
                    CtrlList2.PadByName("List").Var().Value(p1);
                    CtrlList1.OnselValueChanged();
                    CtrlList2.OnselValueChanged();
                    break;
                case 3: //Controller
                    object p1 = CtrlList2.ChildByIndex(0);
                    CtrlList1.PadByName("List").Var().Value(p1);
                    CtrlList2.PadByName("List").Var().Value(p1);
                    CtrlList1.OnselValueChanged();
                    CtrlList2.OnselValueChanged();
                    break;
                default:
                    object p1 = CtrlList1.ChildByIndex(0);
                    CtrlList1.PadByName("List").Var().Value(p1);
                    CtrlList2.PadByName("List").Var().Value(p1);
                    CtrlList1.OnselValueChanged();
                    CtrlList2.OnselValueChanged();
                    break;
            }
        }
        else{
            object p1 = CtrlList1.ChildByIndex(0);
            CtrlList1.PadByName("List").Var().Value(p1);
            CtrlList2.PadByName("List").Var().Value(p1);
            CtrlList1.OnselValueChanged();
            CtrlList2.OnselValueChanged();
        }
    }

    void SetNewTexts(object item,int ch,int type,int c1,int c2){
        object g = _TTParameter;
        g.SetText(format("%i",ch+1));
        SetItemGO(item,COL_MIDICHANNEL,g);
        int ctype = type;
        switch(ctype){
            case 0:
                g.SetText("Note Off");
                break;
            case 1:
                g.SetText("Note On");
                break;
            case 2:
                g.SetText("Key Pressure");
                break;
            case 3:
                g.SetText("Controller");
                break;
            case 4:
                g.SetText("Program Change");
                break;
            case 5:
                g.SetText("After Touch");
                break;
            case 6:
                g.SetText("PitchBend");
                break;
            case 7:
                g.SetText("SysEx");
                break;
        }
        SetItemGO(item,COL_MIDICTRLTYPE,g);
        SetItemLData(item,COL_MIDICTRLTYPE,ctype);

        if(ctype<5)
            g.SetText(GetControllerString(ctype,c1));
        else 
            g.SetText("...");
        SetItemGO(item,COL_MIDICTRL1,g);
        if(c2 != 0x70000000){
            if(ctype<5)
                g.SetText(GetControllerString(ctype,c2));
            else
                g.SetText("...");
            SetItemGO(item,COL_MIDICTRL2,g);
        }
    }

    void RetrieveParameters(){
        count(iParamList,0);
        //get the parameters out of the devices preset list
        object presetList = CurrentDevice.ChildByClass("PepPresetList");
        if(presetList != nil)
            iParamCount = presetList.ParameterList(iParamList);
        else
            iParamCount = CurrentDevice.ParameterList(iParamList,true,PUBLIC|RESTOREINPRESET);
        //do some checks wether parameters should not be displayed
        for(int i=0;i<iParamCount;i++){
            if(iParamList[i] != nil) continue;
            for(int j=(i+1);j<iParamCount;j++)
                iParamList[j-1] = iParamList[j];
            iParamCount--;
        }
        if(DevParamList != nil){
            count(DevParamList.Entries,iParamCount+1);
            count(DevParamList.Values,iParamCount+1);
            int i;
            if(SelMIDIManager != nil){
                for(i=0;i<iParamCount;i++){
                    string h = iParamList[i].FullName(CurrentDevice);
                    if(SelMIDIManager.ControllerByParameter(iParamList[i]) != nil)
                        h = h+" (*)";
                    DevParamList.Entries[i] = h;
                    DevParamList.Values[i] = i;
                }
            }
            else{
                for(i=0;i<iParamCount;i++){
                    DevParamList.Entries[i] = iParamList[i].FullName(CurrentDevice);;
                    DevParamList.Values[i] = i;
                }
            }
            DevParamList.Entries[iParamCount] = "No Parameter !";
            DevParamList.Values[iParamCount] = iParamCount;
            DevParamList.OnEntriesChanged();
        }
    }




//DISPATCHER HANDLERS BEGIN HERE
    public void OnNewManager(){
        int newIndex = GetAllManagers();
        if(SelMIDIManager == nil)
            VarByName("MIDIManager").ValueSendCall(0);
    }

    public void OnDeleteManager(){
        int newIndex = GetAllManagers();
        VarByName("MIDIManager").ValueCallSend(newIndex);
    }
/*
    public void OnNewControllerPreset(object midimanager,object preset){
        //CALLED, when a ControllerPreset has been restored
        Preset = preset;
        if(Preset != nil ){
            object info = Preset.InfoByType(PRESETINFO_CTRLTYPE);
            if(info != nil)
                VarByName("FixedChannel").ValueSendCall(info.GetTypeFlags());
        }

        if(SelMIDIManager == midimanager){
            for(int i=0;i<iParamCount;i++){
                string h = iParamList[i].FullName(CurrentDevice);
                if(SelMIDIManager.ControllerByParameter(iParamList[i]) != nil)
                    h = h+" (*)";
                DevParamList.Entries[i] = h;
            }
            DevParamList.OnEntriesChanged();
            FillList();
            VarByName("ParameterIndex").ValueSendCall(0);
        }
    }
*/
    public void OnPresetRestored(){
        for(int i=0;i<iParamCount;i++){
            string h = iParamList[i].FullName(CurrentDevice);
            if(SelMIDIManager.ControllerByParameter(iParamList[i]) != nil)
                h = h+" (*)";
            DevParamList.Entries[i] = h;
        }
        DevParamList.OnEntriesChanged();
        FillList();
        VarByName("ParameterIndex").ValueSendCall(0);
    }

    public int OnMidiChannelChanged(){
        object g = _TTParameter;
        if(MidiChannel < 16)
            g.SetText(format("%i",MidiChannel+1));
        else
            g.SetText("O");
        if(FixedChannel == 0){
            for(int i=0;i<iParamCount;i++){
                object item = FindPData(nil,iParamList[i],0);
                if(item != nil)
                    SetItemGO(item,COL_MIDICHANNEL,g);
            }
			UpdateList( true );
			UpdateScrollbars( false );
            VarByName("ParameterIndex").ValueCallSend(GetInternalParamIdx(SelectedPItem));
        }
    }

    void DisconnectManager(){
        if(SelMIDIManager != nil){
            PadByName("FixedChannel").Disconnect(SelMIDIManager.PadByName("FixedChannel"));
            PadByName("MidiChannel").Disconnect(SelMIDIManager.PadByName("MidiChannel"));
            PadByName("MIn").Disconnect(SelMIDIManager.PadByName("MIn"));
        }
    }

    void ConnectManager(){
        if(SelMIDIManager != nil){
            SelMIDIManager.PadByName("FixedChannel").ConnectVisible(PadByName("FixedChannel"));
            SelMIDIManager.PadByName("MidiChannel").ConnectVisible(PadByName("MidiChannel"));
            SelMIDIManager.PadByName("MIn").ConnectVisible(PadByName("MIn"));
        }
    }

    public void OnMIDIManagerChanged(){
        DisconnectManager();

        if(MIDIManager != -1){
            SelMIDIManager = SApp.MidiManagerByIndex(MIDIManager);
            if(SelMIDIManager != nil){
                ConnectManager();
                for(int i=0;i<iParamCount;i++){
                    string h = iParamList[i].FullName(CurrentDevice);
                    if(SelMIDIManager.ControllerByParameter(iParamList[i]) != nil)
                        h = h+" (*)";
                    DevParamList.Entries[i] = h;
                }
                DevParamList.OnEntriesChanged();
                FillList();
                VarByName("ParameterIndex").ValueSendCall(0);
            }
        }
    }

    public void OnAddControllerChanged(){
        if(ParameterIndex>=count(iParamList)) return;
        if(CType >= 8){
            MessageBox("You have to select a controller type","Could not add a MIDI controller",MB_OK);
            return;
        }
        if(CType == 7){
            MessageBox("SysEx Controllers are not implemented for now!","Could not add a MIDI controller",MB_OK);
            return;
        }
        int v;
        v = SetCType(v,CType);
        v = SetChannel(v,CChannel);
        if(CType != 6 && CType != 5){
            if(CCtrl1 == 128){
                MessageBox("You have to specify a controller number","Could not add a MIDI controller",MB_OK);
                return;
            }
            v = SetCNum(v,CCtrl1);
            if(CCtrl2 == 128){
                //adding a non linked controller
                object ctrl = SelMIDIManager.AddController(iParamList[ParameterIndex],v);
                if(ctrl != nil){
                    DevParamList.Entries[ParameterIndex] = iParamList[ParameterIndex].FullName(CurrentDevice) + " (*)";
                    DevParamList.OnEntriesChanged();
                    ctrl.Route2Out(iParamList[ParameterIndex].GetVar());
                    FillList();
                }
                else
                    MessageBox("Adding Controller failed !","Could not add a MIDI controller",MB_OK);
            }
            else{
                //adding a linked controller
                int v2=v;
                SetCNum(v2,CCtrl2);
                object ctrl = SelMIDIManager.AddController(iParamList[ParameterIndex],v,v2);
                if(ctrl != nil){
                    DevParamList.Entries[ParameterIndex] = iParamList[ParameterIndex].FullName(CurrentDevice) + " (*)";
                    DevParamList.OnEntriesChanged();
                    ctrl.Route2Out(iParamList[ParameterIndex].GetVar());
                    FillList();
                }
                else
                    MessageBox("Adding 14 Bit Controller failed !","Could not add a MIDI controller",MB_OK);
            }
        }
        else{
            if(CType == 6){
                //adding pitch
                object ctrl = SelMIDIManager.AddController(iParamList[ParameterIndex],v);
                if(ctrl != nil){
                    DevParamList.Entries[ParameterIndex] = iParamList[ParameterIndex].FullName(CurrentDevice) + " (*)";
                    DevParamList.OnEntriesChanged();
                    ctrl.Route2Out(iParamList[ParameterIndex].GetVar());
                    FillList();
                }
                else
                    MessageBox("Adding Controller failed !","Could not add a MIDI controller",MB_OK);
            }
        }
    }

    public void OnRemControllerChanged(){
        if(ParameterIndex>=count(iParamList)) return;
        SelMIDIManager.RemoveController(iParamList[ParameterIndex]);
        DevParamList.Entries[ParameterIndex] = iParamList[ParameterIndex].FullName(CurrentDevice);
        DevParamList.OnselEntryChanged();
        FillList();
    }

    public void OnNoteBaseChanged(){
        string notes[];
        notes = new string[12];
        notes[0] = "C";
        notes[1] = "C#";
        notes[2] = "D";
        notes[3] = "D#";
        notes[4] = "E";
        notes[5] = "F";
        notes[6] = "F#";
        notes[7] = "G";
        notes[8] = "G#";
        notes[9] = "A";
        notes[10] = "A#";
        notes[11] = "B";
        int curNBase = NoteBase;
        for(int i=0;i<128;i++){
            NoteList.Entries[i] = format("%s%i",notes[i%12],curNBase);
            if((i%12) == 11) curNBase++;
        }
    }    
            
    public void OnCTypeChanged(){
        ChangeListNames();
        if(LearnMode == 0){
            //then we must reconfigure a controller, if there exists one
            object pp = GetIParameter(ParameterIndex);
            object item = FindPData(nil,pp,0);
            if(item != nil){
                object ctrl = SelMIDIManager.ControllerByParameter(pp);
                if(ctrl != nil){
                    if(ctrl.Is14Bit()){
                        int v1,v2;
                        v1 = ctrl.GetCtrl1();
                        v2 = ctrl.GetCtrl2();
                        v1 = SetCType(v1,CType);
                        v2 = SetCType(v2,CType);
                        ctrl.Reconfigure(v1,v2);
                        SetNewTexts(item,CChannel,CType,GetCNum(v1),GetCNum(v2));
                    }
                    else{
                        int v1;
                        v1 = ctrl.GetCtrl1();
                        v1 = SetCType(v1,CType);
                        ctrl.Reconfigure(v1);
                        SetNewTexts(item,CChannel,CType,GetCNum(v1),0x70000000);
                    }
                }
            }
        }
    }

    public void OnCChannelChanged(){
        if(LearnMode == 0){
            //then we must reconfigure a controller, if there exists one
            object pp = GetIParameter(ParameterIndex);
            object item = FindPData(nil,pp,0);
            if(item != nil){
                object ctrl = SelMIDIManager.ControllerByParameter(pp);
                if(ctrl != nil){
                    if(ctrl.Is14Bit()){
                        int v1,v2;
                        v1 = ctrl.GetCtrl1();
                        v2 = ctrl.GetCtrl2();
                        v1 = SetChannel(v1,CChannel);
                        v2 = SetChannel(v2,CChannel);
                        ctrl.Reconfigure(v1,v2);
                        SetNewTexts(item,CChannel,CType,GetCNum(v1),GetCNum(v2));
                    }
                    else{
                        int v1;
                        v1 = ctrl.GetCtrl1();
                        v1 = SetChannel(v1,CChannel);
                        ctrl.Reconfigure(v1);
                        SetNewTexts(item,CChannel,CType,GetCNum(v1),0x70000000);
                    }
                }
            }
        }
    }

    public void OnCCtrl1Changed(){
        if(LearnMode == 0){
            if(CType >= 5){
                VarByName("CCtrl1").ValueSend(128);
                return;
            }
            //then we must reconfigure a controller, if there exists one
            object pp = GetIParameter(ParameterIndex);
            object item = FindPData(nil,pp,0);
            if(item != nil){
                object ctrl = SelMIDIManager.ControllerByParameter(pp);
                if(ctrl != nil){
                    if(ctrl.Is14Bit()){
                        int v1,v2;
                        v1 = ctrl.GetCtrl1();
                        v2 = ctrl.GetCtrl2();
                        v1 = SetCNum(v1,CCtrl1);
                        ctrl.Reconfigure(v1,v2);
                        SetNewTexts(item,CChannel,CType,GetCNum(v1),GetCNum(v2));
                    }
                    else{
                        int v1;
                        v1 = ctrl.GetCtrl1();
                        v1 = SetCNum(v1,CCtrl1);
                        ctrl.Reconfigure(v1);
                        SetNewTexts(item,CChannel,CType,GetCNum(v1),0x70000000);
                    }
                }
            }
        }
    }

    public void OnCCtrl2Changed(){
        if(LearnMode == 0){
            if(CType >= 5){
                VarByName("CCtrl2").ValueSend(128);
                return;
            }
            //then we must reconfigure a controller, if there exists one
            object pp = GetIParameter(ParameterIndex);
            object item = FindPData(nil,pp,0);
            if(item != nil){
                object ctrl = SelMIDIManager.ControllerByParameter(pp);
                if(ctrl != nil){
                    if(ctrl.Is14Bit()){
                        boolean isValid = (CCtrl2 < 128);
                        int v1,v2;
                        v1 = ctrl.GetCtrl1();
                        v2 = ctrl.GetCtrl2();
                        v2 = SetCNum(v2,CCtrl2);
                        if(isValid)
                            ctrl.Reconfigure(v1,v2);
                        else
                            ctrl.Reconfigure(v1);
                        SetNewTexts(item,CChannel,CType,GetCNum(v1),GetCNum(v2));
                    }
                    else{
                        int v1,v2;
                        v1 = ctrl.GetCtrl1();
                        v1 = SetCNum(v1,CCtrl1);
                        v2 = 0;
                        v2 = SetChannel(v2,GetChannel(v1));
                        v2 = SetCType(v2,GetCType(v1));
                        v2 = SetCNum(v2,CCtrl2);
                        if(CCtrl2<128)
                            ctrl.Reconfigure(v1,v2);
                        else
                            ctrl.Reconfigure(v1);
                        SetNewTexts(item,CChannel,CType,GetCNum(v1),GetCNum(v2));
                    }
                }
            }
        }
    }

    public void OnParameterIndexChanged(){
        if(LearnMode != 0){
            if(SelMIDIManager.ControllerByParameter(GetIParameter(ParameterIndex)) == nil){
                OnAddControllerChanged();
                VarByName("LearnMode").ValueSendCall(0);
                return;
            }
        }
        SetDisplayFromParameter(ParameterIndex);
    }

    public void OnLearnModeChanged(){
        if(LearnMode == 1)
            VarByName("CCtrl2").ValueCallSend(128);
    }

    public void OnShowNamesChanged(){
        ChangeListNames();
        int old = -1;
        if(SelectedPItem != nil)
            old = GetNumberByItem(SelectedPItem);
        FillList();
        if(old != -1){
            object item = GetItemByNumber(old);
            if(item != nil){
                SelectItem(item,0);
                ScrollToItem(item);
            }
        }
    }
    
    public void Update() {
        AddToPadRouteCB(PadByName("MIn",true));
        ParameterIndex = -1;
        ManagerList = Parent().ChildByName("ResizeBorder").ChildByName("ControllerManager").ChildByIndex(0);
        DevParamList = Parent().ChildByName("ResizeBorder").ChildByName("DevParam").ChildByIndex(0);
        NoteList = Parent().ChildByName("ResizeBorder").ChildByName("NotesList");
        CtrlList1 = Parent().ChildByName("ResizeBorder").ChildByName("Controller 1");
        CtrlList2 = Parent().ChildByName("ResizeBorder").ChildByName("Controller 2");
        object App = SApp.GetModule();
        if(App!=nil) {
            object obj = App.ChildByClass("InfoModule");
            obj.VarByName("SelectedModule").PadByModule().Connect(VarByName("SelDevice").PadByModule());
        }
        GetAllManagers();
        if(count(ManagerList.Entries)>0)
            VarByName("MIDIManager").ValueSendCall(0);
        else
            VarByName("MIDIManager").ValueSendCall(-1);
		super.Update();
        OnNoteBaseChanged();
        OnShowNamesChanged();
    }

	public void InitCols() {
		SetColumn( COL_PARAMETER,		    120, 2, 0);
        SetColumn( COL_MIDICHANNEL,         20,  2, 2);
		SetColumn( COL_MIDICTRLTYPE,		120, 2, 0 );
		SetColumn( COL_MIDICTRL1,			120, 2, 0 );
		SetColumn( COL_MIDICTRL2,	        120, 2, 0 );
		SetDefaultColumn( COL_PARAMETER );
		DefaultPData( nil );
		SetColumnSelection(1);
		SetStyle( GetStyle() | 0x40000 );
	}

	public void OnSelDeviceChanged() {
        int manIdx = -1;
        if(Freeze == 0){
            CurrentDevice = SelDevice;
            if(CurrentDevice != nil){
                RetrieveParameters();
                ModuleName = CurrentDevice.Name();
                PepSendPad("ModuleName");
                for(int i=0;i<count(ManagerList.Entries);i++)
                    if(ManagerList.Entries[i]==ModuleName){ manIdx = i; break; }
            }
    	    FillList();
            if(manIdx != -1)
                VarByName("MIDIManager").ValueSendCall(manIdx);
        }
        else if(CurrentDevice == nil){
            Freeze = 0;
            OnSelDeviceChanged();
            Freeze = 1;
        }   
	}

	public object AddToViewTree(object pvi, int vid) {
		ListVI = super.AddToViewTree(pvi, vid);
		FillList();
		return ListVI;
	}

    public void OnMInChanged(){
        if(LearnMode == 0 || CurrentDevice==nil) return;
        int Q = MIn;
        Q = Q & 0x7fffffff;
        Q = Q-(Q&0xff);
        int g = GetCType(Q);
        if(g==7) return;
        if(g==6 || g == 5) Q=Q&0x7fff0000;
        object p = SelMIDIManager.ParameterByControllerID(Q);
        if(p == nil){
            SetCPadsFromValue(Q);
            DeselectItem();
            ChangeListNames();
            VarByName("ParameterIndex").ValueSend(iParamCount);
            VarByName("CChannel").Send();
            VarByName("CType").Send();
            VarByName("CCtrl1").Send();
            VarByName("CCtrl2").Send();
        }
        else{
            int idx = GetInternalParamIdx(FindPData(nil,p,0));
            SetDisplayFromParameter(idx);
            VarByName("ParameterIndex").ValueSend(idx);
        }
    }

	void FillList() {
		if(TopVI()!=nil && SelMIDIManager != nil) {
			RemoveAll();
			if(CurrentDevice!=nil) {
				object g, r, p, v;
				for(int i=0;i<iParamCount;i++) {
					p = iParamList[i];
                    object ctrl = SelMIDIManager.ControllerByParameter(p);
                    if(ctrl != nil){
					    v = p.GetVar();
					    if(v == nil) {
						    g = _TTNormal;
						    g.SetText("<not assigned>");
					    } else {
						    g = _TTParameter;
						    g.SetText(p.FullName(CurrentDevice));
					    }

					    r = SetItemGO( nil, COL_PARAMETER, g, 0, p);
                        SetItemPData(r,COL_PARAMETER, p);
                        int no = ctrl.GetCtrl1();
                        g.SetText(format("%i",GetChannel(no)+1));
                        SetItemGO(r,COL_MIDICHANNEL,g);
                        int ctype = GetCType(no);
                        switch(ctype){
                            case 0:
                                g.SetText("Note Off");
                                break;
                            case 1:
                                g.SetText("Note On");
                                break;
                            case 2:
                                g.SetText("Key Pressure");
                                break;
                            case 3:
                                g.SetText("Controller");
                                break;
                            case 4:
                                g.SetText("Program Change");
                                break;
                            case 5:
                                g.SetText("After Touch");
                                break;
                            case 6:
                                g.SetText("PitchBend");
                                break;
                            case 7:
                                g.SetText("SysEx");
                                break;
                        }
                        SetItemGO(r,COL_MIDICTRLTYPE,g);
                        SetItemLData(r,COL_MIDICTRLTYPE,ctype);

                        if(ctype<5)
                            g.SetText(GetControllerString(ctype,GetCNum(no)));
                        else 
                            g.SetText("...");
                        SetItemGO(r,COL_MIDICTRL1,g);
                        SetItemPData(r,COL_MIDICTRL1,ctrl);
                        if(ctrl.Is14Bit()){
                            no = ctrl.GetCtrl2();
                            if(ctype<5)
                                g.SetText(GetControllerString(ctype,GetCNum(no)));
                            else
                                g.SetText("...");
                            SetItemGO(r,COL_MIDICTRL2,g);
                            SetItemPData(r,COL_MIDICTRL2,ctrl.GetLinkedController());
                        }
                        else{
                            g.SetText("...");
                            SetItemGO(r,COL_MIDICTRL2,g);
                            SetItemPData(r,COL_MIDICTRL2,nil);
                        }

                    }
				}
			}
			UpdateList( true );
			UpdateScrollbars( false );  
		}
	}

	void ToggleButton(object r, int col) {
		if( r != nil ) {
			object par;
			GetItemPData( r, COL_PARAMETER, par);
			boolean ret;
			boolean ok = true;

			switch( col ) {
				default: ok = false; break;
			}
			if(ok) {
				if(ret)	_BTCheckBoxOK.Show();
				else	_BTCheckBoxOK.Hide();
				SetItemGO( r, col, _BTCheckBox );
			}
		}
	}

	void SetButton(object r, int col, boolean b) {
		if( r != nil ) {
			object par;
			GetItemPData( r, COL_PARAMETER, par);
			boolean ret;
			boolean ok = true;

			switch( col ) {
				default: ok = false; break;
			}
			if(ok) {
				if(ret)	_BTCheckBoxOK.Show();
				else	_BTCheckBoxOK.Hide();
				SetItemGO( r, col, _BTCheckBox );
			}
		}
	}

    public boolean IsSelectedEditable()
    {
		if(!super.IsSelectedEditable()) 
			return false;
        return SelectedColumn!=COL_PARAMETER;
    }

    public void OnSubItemChanged( object rItem, int nColumn ) 
	{
		object par;
		object go = GetItemGO( rItem, nColumn );
//		GetItemPData( rItem, COL_PARAMETER, par);
		switch(nColumn) {
			case COL_PARAMETER:
				break;
			case COL_MIDICHANNEL:
				break;
			case COL_MIDICTRLTYPE:
				break;
			case COL_MIDICTRL1:
				break;
			case COL_MIDICTRL2:
				break;

		}
    }

	public int OnKeyDown(object event) {
		switch(event.Key()) {
			case KEY_SPACE :	ToggleButton(SelectedPItem, SelectedColumn); return 1;
            case KEY_RETURN:
                    if(LearnMode != 0){
                        OnAddControllerChanged();
                        VarByName("LearnMode").ValueSendCall(0);
                    }
                    return 1;
			case KEY_DELETE :	
					if(SelectedPItem==nil) 
						return 0;
					object par;
					GetItemPData( SelectedPItem, COL_PARAMETER, par);
					SelMIDIManager.RemoveController( par );
                    FillList();
					return 1;
		}
		return super.OnKeyDown(event);
	}

	public int OnLButtonDown(object event) {
		int r = super.OnLButtonDown(event);		
		ToggleButton(SelectedPItem, SelectedColumn);
		return r;
	}

	public int OnMouseMove(object event) {
		super.OnMouseMove(event);

		if(event.LDown() && SelectedPItem!=nil) {
			object r = ItemHitTest( event.X(), event.Y(), 1);
			if(r!=SelectedPItem) {
				object par;
				GetItemPData( SelectedPItem, COL_PARAMETER, par);
				SApp.SetDragObject(par);
			}
		}
		return 1;
	}

    public int OnQueryDrop(object event) {
		object pad;
		SApp.GetDragObject(pad);
		if(CurrentDevice!=nil && pad!=nil && (pad.isPad() || pad.isVar())) {
			ChangeSelection( ItemHitTest( event.X(), event.Y(), 1 ), 1 );
			return SelectedPItem==nil ? 0 : 1;
		}

		return 0;
	}

    public int OnDrop(object event) {
		object pad, var;
		SApp.GetDragObject(pad);
		string name;
		
        if(pad.isVar())
			var=pad;
		else if(pad.isPad())
			var=pad.Var();

        if(var == nil) return 1;
		name = pad.Name();

		object par;
		GetItemPData( SelectedPItem, COL_PARAMETER, par);

		par.SetVar(var);
		FillList();
		return 1;
	}

	void EnableAll(boolean b) {		
		for(object r = GetHeadItem(); r!=nil; r = GetNextItem(r))
			SetButton(r, SelectedColumn, b);
	}

    public int OnQueryContextMenu(object viMe,object viHit) {        
		SelectedPData = nil;
        return CMC_ME;
    }

    public int OnQueryContextEntry( int nIndex, int nSubIndex, string strText, object viMe, object viHit)
    {
        nIndex++;
		switch(nIndex) {
		case 0:	strText = "Enable All"; return SelectedColumn >= COL_PUBLIC && SelectedColumn <= COL_STOREINSUBPRESET ? CM_ENTRY : CM_IGNORE;
			case 1:	strText = "Disable All"; return SelectedColumn >= COL_PUBLIC && SelectedColumn <= COL_STOREINSUBPRESET ? CM_ENTRY : CM_IGNORE;
			case 2:	strText = "Create from dump"; return CM_ENTRY;
		}
		return CM_END;
	}
	
    public void OnContextMenu( int nIndex, int nSubIndex, object viMe, object viHit) {
		switch(nIndex) {
			case 0:	EnableAll(true); break;
			case 1:	EnableAll(false); break;
			case 2:	createFromDump(); break;
		}
	}

	public void OnSelectionChanged() {
        if(LearnMode == 0)
            VarByName("ParameterIndex").ValueCallSend(GetInternalParamIdx(SelectedPItem));
	}

	public void OnParameterChanged() {
    		object r = FindPData( nil, Parameter, COL_PARAMETER );
	    	if(r!=nil)
		    	ChangeSelection(r,0);
	}


	public void OnSortColumnChanged() {
		SetDefaultColumn(SortColumn);
		switch(SortColumn) {
			case COL_PARAMETER:			
			case COL_MIDICHANNEL:		
			case COL_MIDICTRLTYPE:	SortList(3); break;
			default:			SortList(4); break;
		}
		UpdateList();
	}
}
